class DynamicSelect {
  constructor(element, options = {}) {
    let defaults = {
      placeholder: "Select an option",
      columns: 1,
      name: "",
      width: "",
      height: "",
      data: [],
      onChange: function () {},
      countryToCurrencyMap: {
        "united states": "usd",
        "united arab emirates": "aed",
        india: "inr",
      },
    };
    this.currencySymbols = {
      USD: "$",
      AED: "AED",
      INR: "₹",
    };
    this.exchangeRates = {
      USD: 1,
      AED: 3.69,
      INR: 83.55,
    };
    this.options = Object.assign(defaults, options);
    this.selectElement = element;
    if (!this.selectElement) {
      console.error("Element not found");
      return;
    }
    this.countryToCurrencyMap = this.options.countryToCurrencyMap;
    for (const prop in this.selectElement.dataset) {
      if (this.options[prop] !== undefined) {
        this.options[prop] = this.selectElement.dataset[prop];
      }
    }
    this.name = this.selectElement.getAttribute("name")
      ? this.selectElement.getAttribute("name")
      : "dynamic-select-" + Math.floor(Math.random() * 1000000);
    if (!this.options.data.length) {
      let options = this.selectElement.querySelectorAll("option");
      for (let i = 0; i < options.length; i++) {
        this.options.data.push({
          value: options[i].value,
          text: options[i].innerHTML,
          img: options[i].getAttribute("data-img"),
          selected: options[i].selected,
          html: options[i].getAttribute("data-html"),
          imgWidth: options[i].getAttribute("data-img-width"),
          imgHeight: options[i].getAttribute("data-img-height"),
        });
      }
    }
    this.element = this._template();
    this.selectElement.replaceWith(this.element);
    this._updateSelected();
    this._eventHandlers();
    this._fetchGeoPluginData();
    this._fetchExchangeRates();
  }

  async _fetchExchangeRates() {
    try {
      // const response = await fetch(
      //   "https://api.exchangerate-api.com/v4/latest/USD"
      // // https://open.er-api.com/v6/latest/USD
      // );
      // const data = await response.json();
      // this.exchangeRates = data.rates;
      console.log(this.exchangeRates);
      // this.setCurrencySelection(this.countryName);
      this.updatePrices();
    } catch (error) {
      console.error("Error fetching exchange rates:", error);
    }
  }

  async _fetchGeoPluginData() {
    const country = localStorage.getItem("countryName");
    if (!country) {
      try {
        const response = await fetch(
          "https://dev-api-ecommerce-customer.knothost.com/api/geolocation"
        );
        const data = await response.json();
        this.countryName = data.data.geoplugin_countryName.toLowerCase();
        localStorage.setItem("countryName", this.countryName);
      } catch (error) {
        console.error("Error fetching geoplugin data:", error);
      }
    } else {
      // console.log(country);
      this.countryName = country;
    }

    // Set the selected option in the select element
    this.setCurrencySelection(this.countryName);
  }

  setCurrencySelection(countryName) {
    const currencyCode = this.countryToCurrencyMap[countryName] || "usd";
    // console.log('Currency Code:', currencyCode);
    const dynamicSelectOptions = this.element.querySelectorAll(
      ".dynamic-select-option"
    );
    // console.log('Dynamic Select Options:', dynamicSelectOptions);
    dynamicSelectOptions.forEach((option) => {
      // console.log(option.dataset.value, currencyCode);
      if (option.dataset.value === currencyCode) {
        option.classList.add("dynamic-select-selected");
        this.element.querySelector(".dynamic-select-header").innerHTML =
          option.innerHTML;
        this.element.querySelector("input").value =
          option.getAttribute("data-value");
        this.data.forEach((data) => (data.selected = false));
        this.data.filter(
          (data) => data.value == option.getAttribute("data-value")
        )[0].selected = true;
        this.element
          .querySelector(".dynamic-select-header")
          .classList.remove("dynamic-select-header-active");
        this.options.onChange(
          option.getAttribute("data-value"),
          option.querySelector(".dynamic-select-option-text")
            ? option.querySelector(".dynamic-select-option-text").innerHTML
            : "",
          option
        );
        // this.updatePrices(); // Update prices when an option is clicked

        // Set countryName based on the selected option
        const selectedCountry = Object.keys(this.countryToCurrencyMap).find(
          (country) =>
            this.countryToCurrencyMap[country] ===
            option.getAttribute("data-value")
        );
        this.countryName = selectedCountry;
        localStorage.setItem("countryName", selectedCountry);
      } else {
        option.classList.remove("dynamic-select-selected");
      }
    });
    if (this.exchangeRates) {
      this.updatePrices();
    }
  }

  updatePrices() {
    if (this.exchangeRates && Object.keys(this.exchangeRates).length > 0) {
      const selectedCurrency = this.selectedValue.toUpperCase();
      localStorage.setItem("countryName", this.countryName);
      const priceElements = document.querySelectorAll(".price");
      const website_type = document.getElementById("website_type");
      const budget = document.getElementById("budget");
      const budget2 = document.getElementById("budget2");
      if (budget) budget.style.display = "none";
      if (website_type) website_type.selectedIndex = 0;
      // console.log(this.exchangeRates)
      console.log(this.currencySymbols[selectedCurrency], selectedCurrency);
      priceElements.forEach((priceElement) => {
        const initialPriceUSD = parseFloat(priceElement.dataset.initialPrice);
        const convertedPrice =
          initialPriceUSD * this.exchangeRates[selectedCurrency];
        priceElement.textContent =
          this.currencySymbols[selectedCurrency] +
          " " +
          convertedPrice.toFixed(2);
        const dataDiv = document.querySelector(".data-div");
        dataDiv.setAttribute(
          "data-exchange-rates",
          this.exchangeRates[selectedCurrency]
        );
        dataDiv.setAttribute(
          "data-selected-currency",
          this.currencySymbols[selectedCurrency]
        );
      });
      if (budget2) {
        budget2.innerHTML = `
            <option value="">Your Budget</option>
            <option value="0 - 100 USD">${
              this.currencySymbols[selectedCurrency]
            }${(0 * this.exchangeRates[selectedCurrency]).toFixed(2)} - ${
          this.currencySymbols[selectedCurrency]
        }${(100 * this.exchangeRates[selectedCurrency]).toFixed(2)} </option>
            <option value="100 - 500 USD">${
              this.currencySymbols[selectedCurrency]
            }${(100 * this.exchangeRates[selectedCurrency]).toFixed(2)} - ${
          this.currencySymbols[selectedCurrency]
        }${(500 * this.exchangeRates[selectedCurrency]).toFixed(2)} </option>
            <option value="501 - 1000 USD">${
              this.currencySymbols[selectedCurrency]
            }${(501 * this.exchangeRates[selectedCurrency]).toFixed(2)} - ${
          this.currencySymbols[selectedCurrency]
        }${(1000 * this.exchangeRates[selectedCurrency]).toFixed(2)} </option>
        `;
      }
    }
  }

  _template() {
    let optionsHTML = "";
    for (let i = 0; i < this.data.length; i++) {
      let optionWidth = 100 / this.columns;
      let optionContent = "";
      if (this.data[i].html) {
        optionContent = this.data[i].html;
      } else {
        optionContent = `
                    ${
                      this.data[i].img
                        ? `<img src="${this.data[i].img}" alt="${
                            this.data[i].text
                          }" class="${
                            this.data[i].imgWidth && this.data[i].imgHeight
                              ? "dynamic-size"
                              : ""
                          }" style="${
                            this.data[i].imgWidth
                              ? "width:" + this.data[i].imgWidth + ";"
                              : ""
                          }${
                            this.data[i].imgHeight
                              ? "height:" + this.data[i].imgHeight + ";"
                              : ""
                          }">`
                        : ""
                    }
                    ${
                      this.data[i].text
                        ? '<span class="dynamic-select-option-text">' +
                          this.data[i].text +
                          "</span>"
                        : ""
                    }
                `;
      }
      optionsHTML += `
                <div class="dynamic-select-option${
                  this.data[i].value == this.selectedValue
                    ? " dynamic-select-selected"
                    : ""
                }${
        this.data[i].text || this.data[i].html ? "" : " dynamic-select-no-text"
      }" data-value="${this.data[i].value}" style="width:${optionWidth}%;${
        this.height ? "height:" + this.height + ";" : ""
      }">
                    ${optionContent}
                </div>
            `;
    }
    let template = `
            <div class="dynamic-select ${this.name}"${
      this.selectElement.id ? ' id="' + this.selectElement.id + '"' : ""
    } style="${this.width ? "width:" + this.width + ";" : ""}${
      this.height ? "height:" + this.height + ";" : ""
    }">
                <input type="hidden" name="${this.name}" value="${
      this.selectedValue
    }">
                <div class="dynamic-select-header" style="${
                  this.width ? "width:" + this.width + ";" : ""
                }${
      this.height ? "height:" + this.height + ";" : ""
    }"><span class="dynamic-select-header-placeholder">${
      this.placeholder
    }</span></div>
                <div class="dynamic-select-options" style="${
                  this.options.dropdownWidth
                    ? "width:" + this.options.dropdownWidth + ";"
                    : ""
                }${
      this.options.dropdownHeight
        ? "height:" + this.options.dropdownHeight + ";"
        : ""
    }">${optionsHTML}</div>
            </div>
        `;
    let element = document.createElement("div");
    element.innerHTML = template;
    return element;
  }

  _eventHandlers() {
    // Handle click events on dynamic select options
    this.element
      .querySelectorAll(".dynamic-select-option")
      .forEach((option) => {
        option.onclick = () => {
          this.element
            .querySelectorAll(".dynamic-select-selected")
            .forEach((selected) =>
              selected.classList.remove("dynamic-select-selected")
            );
          option.classList.add("dynamic-select-selected");
          this.element.querySelector(".dynamic-select-header").innerHTML =
            option.innerHTML;
          this.element.querySelector("input").value =
            option.getAttribute("data-value");
          this.data.forEach((data) => (data.selected = false));
          this.data.filter(
            (data) => data.value == option.getAttribute("data-value")
          )[0].selected = true;
          this.element
            .querySelector(".dynamic-select-header")
            .classList.remove("dynamic-select-header-active");
          this.options.onChange(
            option.getAttribute("data-value"),
            option.querySelector(".dynamic-select-option-text")
              ? option.querySelector(".dynamic-select-option-text").innerHTML
              : "",
            option
          );
          this.updatePrices(); // Update prices when an option is clicked

          // Set countryName based on the selected option
          const selectedCountry = Object.keys(this.countryToCurrencyMap).find(
            (country) =>
              this.countryToCurrencyMap[country] ===
              option.getAttribute("data-value")
          );
          this.countryName = selectedCountry;
          localStorage.setItem("countryName", selectedCountry);
        };
      });

    // Handle click events on dynamic select header
    this.element.querySelector(".dynamic-select-header").onclick = () => {
      this.element
        .querySelector(".dynamic-select-header")
        .classList.toggle("dynamic-select-header-active");
    };

    // Handle click events on labels associated with the original select boxes
    if (
      this.selectElement.id &&
      document.querySelector('label[for="' + this.selectElement.id + '"]')
    ) {
      document.querySelector(
        'label[for="' + this.selectElement.id + '"]'
      ).onclick = () => {
        this.element
          .querySelector(".dynamic-select-header")
          .classList.toggle("dynamic-select-header-active");
      };
    }

    // Handle change events on the original select box
    this.selectElement.addEventListener("change", () => {
      const selectedOption =
        this.selectElement.options[this.selectElement.selectedIndex];
      const dynamicOption = this.element.querySelector(
        `[data-value="${selectedOption.value}"]`
      );
      if (dynamicOption) {
        dynamicOption.click();
      }
    });

    // Handle click events outside the dynamic select
    document.addEventListener("click", (event) => {
      if (
        !event.target.closest("." + this.name) &&
        !event.target.closest('label[for="' + this.selectElement.id + '"]')
      ) {
        this.element
          .querySelector(".dynamic-select-header")
          .classList.remove("dynamic-select-header-active");
      }
    });
  }

  _updateSelected() {
    if (this.selectedValue) {
      this.element.querySelector(".dynamic-select-header").innerHTML =
        this.element.querySelector(".dynamic-select-selected").innerHTML;
    }
  }

  get selectedValue() {
    let selected = this.data.filter((option) => option.selected);
    selected = selected.length ? selected[0].value : "";
    return selected;
  }

  set data(value) {
    this.options.data = value;
  }

  get data() {
    return this.options.data;
  }

  set selectElement(value) {
    this.options.selectElement = value;
  }

  get selectElement() {
    return this.options.selectElement;
  }

  set element(value) {
    this.options.element = value;
  }

  get element() {
    return this.options.element;
  }

  set placeholder(value) {
    this.options.placeholder = value;
  }

  get placeholder() {
    return this.options.placeholder;
  }

  set columns(value) {
    this.options.columns = value;
  }

  get columns() {
    return this.options.columns;
  }

  set name(value) {
    this.options.name = value;
  }

  get name() {
    return this.options.name;
  }

  set width(value) {
    this.options.width = value;
  }

  get width() {
    return this.options.width;
  }

  set height(value) {
    this.options.height = value;
  }

  get height() {
    return this.options.height;
  }
}

document
  .querySelectorAll("[data-dynamic-select]")
  .forEach((select) => new DynamicSelect(select));
